#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <strings.h>
#include <fcntl.h>
#include <poll.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <errno.h>
#include <arpa/inet.h>

int
sendall (int fd, void *buf, int *len)
{
  int sent_len = 0;
  uint8_t *b = buf;
  int n;

  while (sent_len < *len) {
    n = write(fd, b + sent_len, *len - sent_len);
    if (n < 0) {
      break;
    }
    sent_len += n;
  }

  if (n < 0) {
    *len = sent_len;
    return -1;
  }

  return 0;
}

int
open_tcp_listener (const char *port)
{
  struct addrinfo hints, *addrinfo, *ai;
  int n, srv, ret;
  char ipstr[INET6_ADDRSTRLEN];

  bzero(&hints, sizeof(hints));
  hints.ai_family = AF_UNSPEC;
  hints.ai_socktype = SOCK_STREAM;
  hints.ai_flags = AI_PASSIVE;

  if ((n = getaddrinfo(NULL, port, &hints, &addrinfo)) != 0) {
    fprintf(stderr, "getaddrinfo: %s\n", gai_strerror(n));
    exit(1);
  }

  for (ai = addrinfo; ai != NULL; ai = ai->ai_next) {
    switch (ai->ai_family) {
    case AF_INET:
      inet_ntop(ai->ai_family, &((struct sockaddr_in *)ai->ai_addr)->sin_addr,
                ipstr, sizeof(ipstr));
      break;
    case AF_INET6:
      inet_ntop(ai->ai_family, &((struct sockaddr_in6 *)ai->ai_addr)->sin6_addr,
                ipstr, sizeof(ipstr));
      break;
    default: continue;
    }
    printf("Trying to listen on %s on port %s…\n", ipstr, port);

    if ((srv = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol)) < 0) {
      perror("socket");
    }
    if (setsockopt(srv, SOL_SOCKET, SO_REUSEADDR, &(int){1}, sizeof(int)) < 0) {
      perror("setsockopt");
      exit(1);
    }
    if (fcntl(srv, F_SETFL, O_NONBLOCK) < 0) {
      perror("fcntl");
      exit(1);
    }
    if (bind(srv, ai->ai_addr, ai->ai_addrlen) < 0) {
      perror("bind");
      exit(1);
    }
    if (listen(srv, 8) < 0) {
      perror("listen");
      exit(1);
    }
    ret = srv;
    printf("Listening on %s on port %s\n", ipstr, port);
    break;
  }

  freeaddrinfo(addrinfo);

  return ret;
}

void
identify_client (struct sockaddr *addr, size_t addrlen) {

  char host[1024], service[32];
  int status;
  char ipstr[INET6_ADDRSTRLEN];

  if ((status = getnameinfo(addr, addrlen,
                            host, sizeof(host),
                            service, sizeof(service),
                            0)) != 0) {
    fprintf(stderr, "getnameinfo: %s\n", gai_strerror(status));
  }
  inet_ntop(addr->sa_family, addr, ipstr, sizeof(ipstr));
  printf("> %s (%s) from port %s\n", host, ipstr, service);
}

#define MAX_CLIENTS 32

int
main (int argc, char *argv[])
{
  int srv, fd;
  struct sockaddr_storage addr;
  int addrlen;
  struct pollfd fds[MAX_CLIENTS];
  int nfds, i, j, n, len;
  uint8_t buf[2048];

  if (argc != 2) {
    fprintf(stderr, "Usage: %s port\n", argv[0]);
    exit(1);
  }

  if ((srv = open_tcp_listener(argv[1])) < 0) {
    fprintf(stderr, "Could not listen on either IPv4 or IPv6\n.");
    exit(1);
  }

  bzero(fds, sizeof(fds));
  fds[0].fd = srv;
  fds[0].events = POLLIN;
  nfds = 1;

  for (;;) {
    if ((n = poll(fds, nfds, 300000)) < 0) {
      perror("poll");
      continue;
    }
    else if (n == 0) { /* timeout */
      if (nfds == 1) { /* pas de client pendant 5 minutes, on quitte */
        break;
      }
    }
    if (fds[0].revents & POLLIN) { /* nouveau client */
      if ((fd = accept(srv, (struct sockaddr *)&addr, &addrlen)) < 0) {
        if (errno == EWOULDBLOCK) {
          fprintf(stderr, "accept EWOULDBLOCK");
        }
      }
      fprintf(stderr, "New connection with fd = %d\n", fd);
      identify_client((struct sockaddr *)&addr, addrlen);

      if (nfds == MAX_CLIENTS) { /* déjà plein */
        write(fd, "Sorry, server full!", 20);
        fprintf(stderr, "Server full, closing connection with fd = %d\n", fd);
        close(fd);
      }
      fds[nfds].fd = fd;
      fds[nfds].events = POLLIN;
      nfds++;
    }
    for (i = 1; i < nfds; i++) {
      if ((fds[i].revents & POLLIN) == 0) continue;
      fprintf(stderr, "Message from client with fd = %d\n", fds[i].fd);
      bzero(buf, sizeof(buf));
      if ((len = read(fds[i].fd, buf, sizeof(buf))) < 0) {
        perror("read");
      }
      else if (len == 0) { /* déconnexion */
        fprintf(stderr, "Disconnect from client with fd = %d\n", fds[i].fd);
        close(fds[i].fd);
        for (j = i + 1; j < nfds; j++) { /* recompacter fds */
          fds[j-1].fd = fds[j].fd;
        }
        nfds--;
        i--;
      }
      else { /* nouveau message du client */
        for (j = 1; j < nfds; j++) {
          if (fds[j].fd == fds[i].fd) continue;
          n = len;
          if (sendall(fds[j].fd, buf, &len) < 0) {
            fprintf(stderr, "sendall: only sent %d of %d bytes\n", len, n);
          }
        }
      }
    }
  }

  close(srv);

  return 0;
}
